import argparse
import os

from src.Injector.img_injector import ImageInjection
from src.Injector.pdf_injector import PDFInjection
from src.utils import load_yaml_config, setup_logging

def parse_arguments():
    parser = argparse.ArgumentParser(description='Welcome to the Blueinfy File Injector CLI')
    
    # Common arguments
    parser.add_argument('--config_file', type=str, default="config.yaml", help='Enter the config file path')
    parser.add_argument('--mode', type=str, choices=['pdf', 'img'], required=True, help='Select mode: pdf or img')
    parser.add_argument('--file', type=str, required=True, help='Input file path')
    parser.add_argument('--prompt_file', type=str, required=True, help='Input txt file containing prompt')
    parser.add_argument('--output_file', type=str, required=True, help='Output file path')
    parser.add_argument('--num_times', type=int, default=1, help='Number of times to inject prompt')
    parser.add_argument('--font_size', type=int, default=20, help='Font size for text injection')
    
    # Optional arguments specific to image mode
    img_args = parser.add_argument_group('Image mode arguments')
    img_args.add_argument('--image_mode', type=str, default='RGB', choices=['L', 'RGB'], help='Image mode (L or RGB)')
    img_args.add_argument('--method', type=str, default='blend', choices=['blend', 'color'], help='Text injection method')
    img_args.add_argument('--color', type=str, default='black', help='Specify color only if method="color". Specify either as tuple or string (eg: white) or as hex value')
    img_args.add_argument('--pixel_diff', type=int, default=80, choices = range(256), help='Specify color only if method="blending". Specify the difference in pixels you want while blending, default is 80, Range is 0 to 255')
    return parser.parse_args()

def main():

    logger = setup_logging()
    args = parse_arguments()
    
    if not os.path.exists(args.file):
        raise Exception (f"Input File {args.file} does not exist")
    
    if not os.path.exists(args.prompt_file):
        raise Exception (f"Prompts File {args.prompt_file} does not exist")
    
    file_type = args.file.split(".")[-1]

    if file_type == "pdf":
        mode = "pdf"
    elif file_type in ["jpg", "png", "jpeg"]:
        mode = "img"
    else:
        raise Exception(f"File type {file_type} is not accepted - give either jpg/png/jpeg files as input")
    
    logger.info(f"Mode selected: {mode}")

    with open(args.prompt_file, "r", encoding="utf-8") as file:
        prompt = file.read()
    logger.info(f"Read input prompt from file: {args.prompt_file}")

    output_file = args.output_file

    while os.path.exists(output_file):
        choice = input(f"The file {output_file} already exists. \nDo you want to overwrite the file (y/n): ").lower()

        if choice=="n":
            output_file = input(f"Please enter the new file name: ")
        elif choice=="y":
            break
        else:
            logger.error(f"Invalid choice detected - please enter either 'y' or 'n' as input")
    
    prompt = prompt + " "
    prompt = prompt * args.num_times

    if mode == "pdf":
        try:
            config = load_yaml_config(yaml_path=args.config_file)
            gen = PDFInjection(file=args.file, config=config, src="cli")
            gen.addText(prompt)
            # We can add multiple addText calls as below
            # gen.addText("Hello!", page_number=1, color="red") 
            # gen.addText("PDF!", x_coordinate=10, y_coordinate=200, page_number=2)
            gen.generate(output_file)
            logger.info(f"Prompt successfully injected in file: {output_file}")

        except Exception as e:
            logger.error(f"Error injecting prompt in file {args.file}: {e}")

    elif mode == "img":
        
        try:
            injector = ImageInjection(image_path=args.file, image_mode=args.image_mode, font_size=args.font_size)
            logger.info(f"Image Injector Initialized")
            image_stream = injector.inject_text(text=prompt, method=args.method, color=args.color, pixel_diff=args.pixel_diff)
            with open(output_file, "wb") as f:
                f.write(image_stream.getvalue())

            logger.info(f"Prompt successfully injected in file: {output_file}")
        
        except Exception as e:
            logger.error(f"Error injecting prompt in file {args.file}: {e}")
        
        finally:
            if image_stream:
                image_stream.close()
                logger.info("Image stream closed")
            else:
                logger.info("No image stream to close")

if __name__ == "__main__":
    main()