from io import BytesIO
from PIL import Image, ImageDraw, ImageFont

from src.utils import setup_logging

class ImageInjection:
    
    def __init__(self, image_path, font_size=5, image_mode="RGB", pixel_diff=80):
        self.image_path = image_path
        self.image_mode = image_mode
        self.image = Image.open(image_path).convert(image_mode)
        self.draw = ImageDraw.Draw(self.image)
        self.font_size = font_size
        self.font = ImageFont.load_default().font_variant(size=self.font_size)
        self.logger = setup_logging()

    def background_blender(self, x, y, pixel_diff):
        """Returns the text color of a char by blending it with surrounding background"""

        background_color = self.image.getpixel((x, y))

        if self.image_mode == "RGB":
            
            r = (background_color[0] + pixel_diff) % 255
            g = (background_color[1] + pixel_diff) % 255
            b = (background_color[2] + pixel_diff) % 255
            return (r, g, b)
        
        elif self.image_mode == "L":

            if isinstance(background_color, int):
              gray = (background_color + pixel_diff) % 255
            else:
              gray = (ord(background_color) + pixel_diff) % 255
            
            return gray
        
        else:
 
            raise ValueError("Unsupported image mode. Choose 'RGB' or 'L' as image_mode.")


    def inject_text(self, text, method="color", color=(255, 255, 255), pixel_diff=80):
        """Injects text into the image using the specified method and color."""

        if method not in ["color", "blend"]:
            raise ValueError("Invalid method selected. Choose 'color' or 'blend' as method.")

        x = 5
        y = 5
        
        image_bytes = BytesIO() #Initializing Imagebytes object

        for char in text:

            char_bbox = self.font.getbbox(char)
            if x + char_bbox[2] > self.image.width:
                x = char_bbox[2]
                y += self.font.getbbox("hg")[3] 
                if y + char_bbox[3] > self.image.height:
                    self.logger.warning("Text exceeds image boundaries.")
                    self.image.save(image_bytes, format="jpeg")
                    image_bytes.seek(0)
                    return image_bytes

            if method == "color":
                self.draw.text((x, y), char, fill=color, font=self.font)
            
            else:
                color = self.background_blender(x, y, pixel_diff)
                self.draw.text((x, y), char, font=self.font, fill=color)
            
            x += char_bbox[2]

        self.image.save(image_bytes, format="jpeg")
        image_bytes.seek(0)
        return image_bytes