import io

from pypdf import PdfWriter, PdfReader, Transformation
from reportlab.lib.colors import HexColor
from reportlab.pdfgen.canvas import Canvas

class PDFInjection:

    def __init__(self, file, config, src):
        
        self.config = config
        self.packet = io.BytesIO()
        self.load_pdf(file, src)
        self.c = Canvas(self.packet, pagesize=(self.template_page.mediabox.width,
                                               self.template_page.mediabox.height))
        self.output = PdfWriter()
        self.text_packets = []  # List to store text packets (Required to handle multi-edits in pdf)

    def load_pdf(self, file, src):
        try:
            if src=="ui":
                self.template_pdf = PdfReader(file)
            else:
                self.template_pdf = PdfReader(open(file, "rb"))
            self.template_page = self.template_pdf.pages[0]
            self.num_pages = len(self.template_pdf.pages)  # Get total number of pages
        except Exception as e:
            raise ValueError(f"Error loading PDF: {str(e)}")

    def wrap_text(self, text, font_family, font_size):
        
        max_width = self.template_page.mediabox.width - 50
        lines = []
        words = text.split()
        line = ""

        for word in words:
            if self.c.stringWidth(line + " " + word, font_family, font_size) <= max_width:
                if line:
                    line += " " + word
                else:
                    line = word
            else:
                lines.append(line)
                line = word

        if line:
            lines.append(line)

        return lines

    def addText(self, text, **kwargs):

        font_family = kwargs.get('font_family', self.config.get('font_family', "Helvetica-Bold"))
        font_size = kwargs.get('font_size', self.config.get('font_size', 55))
        color = kwargs.get('color', self.config.get('color', "blue"))
        line_height = font_size * 1.5
        page_number = kwargs.get('page_number', self.config.get('page_number', 1))
        if page_number > self.num_pages:
            raise ValueError(f"Invalid page number: {page_number}")
        
        
        if 'page_number' in kwargs:
            self.packet = io.BytesIO()
            self.c = Canvas(self.packet, pagesize=(self.template_pdf.pages[page_number - 1].mediabox.width, self.template_pdf.pages[page_number - 1].mediabox.height))  # If page_number is provided, this re-initializes the Canvas with the given page size

        # Get x and y coordinates, using defaults if None in config
        x_coordinate = kwargs.get('x_coordinate') 
        if x_coordinate is None:
            x_coordinate = self.config.get('x_coordinate')
            if x_coordinate is None:  
                x_coordinate = 10  #Default

        y_coordinate = kwargs.get('y_coordinate')
        if y_coordinate is None:
            y_coordinate = self.config.get('y_coordinate')
            if y_coordinate is None:  
                y_coordinate = self.template_page.mediabox.height - line_height  #Default

        try:
            if isinstance(color, tuple):
                self.c.setFillColorRGB(color[0] / 255, color[1] / 255, color[2] / 255)
            elif color.startswith('#'):
                self.c.setFillColor(HexColor(color))
            else:
                self.c.setFillColor(color)

            self.c.setFont(font_family, font_size)

            lines = self.wrap_text(text, font_family, font_size)

            y = y_coordinate
            for line in lines:
                self.c.drawString(x_coordinate, y, line)
                y -= line_height

            self.c.save()
            self.packet.seek(0)
            self.text_packets.append((self.packet, page_number - 1))  # Storing the packet and page number

        except Exception as e:
            raise ValueError(f"Invalid color format: {str(e)}")

    def merge(self, **kwargs):
        for packet, page_number in self.text_packets:
            result_pdf = PdfReader(packet)
            result = result_pdf.pages[0]

            rotate = kwargs.get('rotate', self.config.get('rotate', 0))
            translate_x = kwargs.get('translate_x', self.config.get('translate_x', 0))
            translate_y = kwargs.get('translate_y', self.config.get('translate_y', 0))

            op = Transformation().rotate(rotate).translate(tx=translate_x, ty=translate_y)
            result.add_transformation(op)

            self.template_pdf.pages[page_number].merge_page(result)

        for i in range(self.num_pages):
            self.output.add_page(self.template_pdf.pages[i])

    def generate(self, dest=None):
        # First write to memory buffer
        self.merge()

        # Compresssing technique
        for page in self.output.pages:
            
            page.compress_content_streams()
            if hasattr(page, '/Resources'):
                page['/Resources'].compress()

        output_buffer = io.BytesIO()
        self.output.write(output_buffer)
        output_buffer.seek(0)
        
        # If destination is provided, then we write buffer to file
        if dest:
            with open(dest, 'wb') as f:
                f.write(output_buffer.getvalue())
        
        return output_buffer