import base64
import os
import streamlit as st
import html

from src.Injector.img_injector import ImageInjection
from src.Injector.pdf_injector import PDFInjection
from src.utils import load_yaml_config

# Defining file paths
config_file_path = os.path.join("config.yaml")
logo_path = os.path.join("src/assets/logo.png")

# Max size to display on UI 
# Note: This limitation is caused due to streamlit not being able to render pdf files > 2 mb [https://discuss.streamlit.io/t/looking-for-a-solution-to-embed-pdf-files-larger-than-1-5mb-without-splitting/66066]
MAX_DISPLAY_SIZE_MB = 1.5

def remove_borders_from_form():
    st.markdown("""
        <style>
            .stForm {
                border: none !important;
                padding: 0 !important;
                border-radius: 0 !important;
                background-color: transparent !important;
            }
            
            .stForm [data-testid="stFormSubmitButton"] {
                border-color: rgb(255, 75, 75);
                color: rgb(255, 75, 75);
            }
        </style>
    """, unsafe_allow_html=True)

def initialize_session_state():
    defaults = {
        'is_processing': False,
        'processing_complete': False,
        'output_prompts': [],
        'error_message': None,
        'mode': None,
        'file_type': "",
        'file_uploaded': False,
        'pdf_base64': "",
        'pdf_bytes': "",
        'prompt': "",
        'injection_method': ""
    }
    
    for key, default_value in defaults.items():
        if key not in st.session_state:
            st.session_state[key] = default_value
    
    st.set_page_config(
        page_title="Blueinfy File Injector",
        page_icon=":computer:",
        layout="wide"
    )
    st.session_state.page_config_set = True
    st.title("Blueinfy File Injector")
    # This code removes borders from form (Required coz we shifted all form inputs outside to render realtime updates to prompt selection)
    remove_borders_from_form()

def reset_output_states():
    st.session_state.error_message = ""

def show_processing_overlay():
    """Show a processing overlay that prevents UI interaction"""
    if st.session_state.is_processing:
        st.markdown(
            """
            <style>

                .stApp [data-testid="stToolbar"],
                .stApp [data-testid="stDecoration"],
                .stApp [data-testid="stSidebar"] > div,
                .stApp .stTextInput,
                .stApp .stNumberInput,
                .stApp .stFileUploader,
                .stApp .stSelectbox,
                .stApp .stTextArea,
                .stApp .stColorPicker > div,
                .stApp [data-testid="stSidebarNav"],
                .stApp .stButton,
                .stApp .stForm {
                    opacity: 0.7 !important;
                    pointer-events: none !important;
                    cursor: not-allowed !important;
                }
                
                .stApp .stSpinner, 
                .stApp .stProgress {
                    opacity: 1 !important;
                    pointer-events: auto !important;
                    position: relative;
                    z-index: 1000;
                }
            </style>
            """,
            unsafe_allow_html=True
        )

def render_sidebar(config):
    with st.sidebar:
        st.image(logo_path, width=300)
        st.title("Configuration")

        # Font settings
        font_options = ["Helvetica-Bold", "Times-Roman", "Courier", "Helvetica"]
        config["font_family"] = st.selectbox("Font Family", font_options, index=0, disabled=st.session_state.is_processing)
        config["font_size"] = st.number_input("Font Size", min_value=1, max_value=72, value=12, disabled=st.session_state.is_processing)
        
        # Color setting
        config["color"] = st.color_picker("Text Color", "#0000FF", disabled=st.session_state.is_processing)

        config["pixel_diff"] = st.number_input("Pixel Difference", min_value=0, max_value=255, value=80, disabled=st.session_state.is_processing, help="If you select method='blending' in image injection, then the font color used will be the color at the place + the pixel difference value")

def display_prompt(prompt):
    sanitized_prompt = html.escape(prompt)
    st.markdown(
        f"""
        Selected Prompt
        <div style="border: 1px solid #ccc;
                    border-radius: 2px;
                    padding: 10px;
                    background-color: #f0f2f6;
                    height: 100px;
                    overflow-y: scroll;
                    user-select: none;
                    cursor: default;
                    word-wrap: break-word;">
            {sanitized_prompt}
        </div>
        """,
        unsafe_allow_html=True
    )

def process_file(file_input, prompt, num_times, default_config):
    try:
        st.session_state.is_processing = True

        if st.session_state.mode=="error":
            st.session_state.error_message = f"File {file_input.type} is not allowed"
            return
        
        if st.session_state.mode != st.session_state.file_type:
            st.session_state.error_message = f"You selected {st.session_state.file_type} but uploaded {st.session_state.mode} file"
            return
            
        if st.session_state.mode == "pdf":
            with st.spinner('Injecting Prompts...'):
                
                gen = PDFInjection(file=file_input, config=default_config, src="ui")
                prompt = st.session_state.prompt + " "
                prompt = prompt * num_times
                gen.addText(prompt)

            with st.spinner('Generating final PDF...'):
                
                pdf_buffer = gen.generate()
                st.session_state.pdf_bytes = pdf_buffer.getvalue()
                st.session_state.pdf_base64 = base64.b64encode(st.session_state.pdf_bytes).decode('utf-8')
            
            st.session_state.processing_complete = True
            
        elif st.session_state.mode == "img":
            with st.spinner('Injecting Prompts...'):
                prompt = st.session_state.prompt + " "
                prompt = prompt * num_times

                injector = ImageInjection(image_path=file_input, image_mode=default_config['image_mode'], font_size=default_config['font_size'])
                image_bytes = injector.inject_text(text=prompt, method=st.session_state.injection_method, color=default_config['color'], pixel_diff=default_config['pixel_diff'])
                st.session_state.image_bytes = image_bytes
                st.session_state.processing_complete = True
            
    except Exception as e:
        st.session_state.is_processing = False
        st.session_state.error_message = str(e)
        st.rerun()
    
    finally:  
        st.session_state.is_processing = False
        st.rerun()

def main():
    initialize_session_state()
        
    if not os.path.exists(logo_path):
        raise FileNotFoundError("Missing the logo file")
    
    default_config = load_yaml_config(config_file_path)

    mode = st.radio(
        "Select the file type:",
        ["Image", "PDF"],
        horizontal=True,
        disabled=st.session_state.is_processing,
        on_change=reset_output_states
    )
    render_sidebar(config=default_config)
    col1, col2 = st.columns([1, 1])

    if mode=="Image":
        st.session_state.file_type = "img"
    else:
        st.session_state.file_type = "pdf"

    with col1:
        # Add image injection method selection
        if st.session_state.file_type == "img":
            method = st.selectbox(
                label="Choose the method of injection", 
                key="injected_method",
                options=["Blending", "Color"],
                disabled=st.session_state.is_processing,
                help="In Blended method, pixel color is calculated near to the adjoining pixel value, In color method, the color specified is used"
            )
            injection_method = "blend" if method=="Blending" else "color"
            st.session_state.injection_method = injection_method

            image_mode = st.selectbox("Please select the image mode", ["RGB", "Grayscale"], index=0, disabled=st.session_state.is_processing)
            default_config["image_mode"] = "L" if image_mode == "Grayscale" else "RGB"
        
        file_input = st.file_uploader("Upload a file", disabled=st.session_state.is_processing)
        if file_input:
            file_type = file_input.type
            if "pdf" in file_type:
                st.session_state.mode = "pdf"
                st.session_state.file_uploaded = True
            elif "jpg" in file_type or "png" in file_type or "jpeg" in file_type:
                st.session_state.mode = "img"
                st.session_state.file_uploaded = True
            else:
                st.session_state.mode = "error"               

        num_times = st.number_input(
            label="Number of times to inject prompt",
            min_value=1,
            max_value=20,
            value=5,
            disabled=st.session_state.is_processing
        )

    with col2:
        prompt = ""

        prompt_input_type = st.selectbox(
            label="Choose prompt input", 
            key="prompt_input",
            options=["Upload a custom txt file", "Write in textbox", "Choose from default options"],
            disabled=st.session_state.is_processing
        )

        if prompt_input_type == "Write in textbox":
            prompt = st.text_area(
                label="Write a prompt here",
                key="prompt_txt_input",
                placeholder="Write the prompt you want to inject in the PDF",
                height=170,
                disabled=st.session_state.is_processing
            )
            st.session_state.prompt = prompt
        
        elif prompt_input_type == "Choose from default options":
            prompt = st.selectbox(
                label="Choose the default prompt input", 
                key="default_prompt_input",
                options=os.listdir("Prompts"),
                help="Choose a custom prompt to inject",
                disabled=st.session_state.is_processing
            )
            with open(f"Prompts/{prompt}", "r", encoding="utf-8") as f:
                prompt = f.read()
            st.session_state.prompt = prompt
            display_prompt(st.session_state.prompt)

        else:
            prompt_file_input = st.file_uploader("Upload the prompt file", type="txt", disabled=st.session_state.is_processing)
            if prompt_file_input:
                prompt = prompt_file_input.getvalue().decode("utf-8")
            st.session_state.prompt = prompt
            display_prompt(st.session_state.prompt)

    with st.form(key="input_form"):

        submit_button = st.form_submit_button(
            "Inject Prompts",
            disabled=st.session_state.is_processing,
            on_click=reset_output_states
        )
    
    if submit_button:
        
        if not file_input:
            st.error("Please upload the input file!")
            return
        
        if not st.session_state.prompt:
            st.error("Please upload/type a valid prompt to be injected")
            return

        st.session_state.is_processing = True
        show_processing_overlay()
        process_file(file_input, prompt, num_times, default_config)

    # Render injected file in UI
    if st.session_state.processing_complete:

        if st.session_state.mode == "pdf":

            st.session_state.pdf_base64 = base64.b64encode(st.session_state.pdf_bytes).decode('utf-8')
            st.success("Processing completed")
            st.download_button(
                label="Download PDF",
                data=st.session_state.pdf_bytes,
                mime="application/pdf",
                file_name="output.pdf",
            )

            # Checking file size before rendering
            file_size_mb = len(st.session_state.pdf_bytes) / (1024 * 1024)
            
            if file_size_mb <= MAX_DISPLAY_SIZE_MB:
                st.markdown("<h2 style='text-align: center;'>Injected PDF Rendering</h2>", unsafe_allow_html=True)
                
                st.markdown(
                    """
                    <div style="display: flex; justify-content: center; align-items: center; flex-direction: column;">
                        <div style="border: 2px solid #000; padding: 10px;">
                            <iframe 
                                src="data:application/pdf;base64,{pdf_data}"
                                width="600",
                                height="600", 
                                style="border: none;",
                            >
                            </iframe>
                        </div>
                    """.format(pdf_data=html.escape(st.session_state.pdf_base64)),
                    unsafe_allow_html=True
                )
            
            else:
                st.warning(f"File size ({file_size_mb:.1f} MB) exceeds display limit ({MAX_DISPLAY_SIZE_MB} MB). Please download to view.")
            
            st.session_state.processing_complete = False

        elif st.session_state.mode == "img":
            
            st.success("Processing completed")
            
            st.download_button(
                label="Download processed image",
                data=st.session_state.image_bytes.getvalue(),
                file_name="processed_image.jpg",
                mime="image/jpg"
            )
            
            ui_col1, ui_col2, ui_col3 = st.columns([1,2,1])
            with ui_col2:
                st.markdown("<h2>Injected Image Rendering</h2>", unsafe_allow_html=True)
                st.image(st.session_state.image_bytes.getvalue(), caption="Processed Image")
            
            
            st.session_state.processing_complete = False
        
    else:
        
        if st.session_state.error_message:
            st.error(st.session_state.error_message)
            st.session_state.error_message = None

if __name__ == "__main__":
    main()