import streamlit as st
from typing import Dict, Tuple, Any

def render_sidebar(logo_path: str, model_info: Dict[str, Any]) -> Tuple[str, str, str]:
    """
    Render sidebar with model configuration options
    
    Args:
        logo_path: Path to logo image
        model_info: Dictionary containing model provider configurations
    
    Returns:
        Tuple containing: 
            model_provider (str): model provider selected
            selected_model (str): model selected
            api_key (str): API key for the platform
    """

    with st.sidebar:
        st.image(logo_path, width=300)
        st.subheader("Configuration")

        model_provider = st.selectbox(
            "Select Model Provider:",
            list(model_info.keys()),
            key="model_provider",
            disabled=st.session_state.is_processing,
        )
        
        selected_model = st.selectbox(
            "Select Model:",
            list(model_info[model_provider]['models']),
            key="selected_model",
            disabled=st.session_state.is_processing,
        )
        
        api_key = st.text_input(
            f"Enter your {model_provider} API key:",
            type="password",
            key=f"{model_provider.lower()}_api_key",
            disabled=st.session_state.is_processing,
        )

        # Model-specific
        provider_config = model_info[model_provider]
        temperature_range= provider_config['temperature_range']
        top_p_range = provider_config['top_p_range']
        max_tokens_range = provider_config['max_tokens_range']

        temperature = st.slider(
            "Select the Temperature:",
            min_value=temperature_range[0],
            max_value=temperature_range[1],
            value=0.7,
            step=0.1,
            key="temperature",
            help="It controls the randomness and creativity of the responses",
            disabled=st.session_state.is_processing,
        )

        top_p = st.slider(
            "Select the Top_p",
            min_value=top_p_range[0],
            max_value=top_p_range[1],
            value=0.7,
            step=0.1,
            key="top_p",
            disabled=st.session_state.is_processing,
            help="It controls the randomness of the output by determining the possible words to consider when generating the next word.",
        )

        max_tokens = st.slider(
            "Select the max tokens limit of the model",
            min_value=max_tokens_range[0],
            max_value=max_tokens_range[1],
            step=100,
            key="max_tokens",
            disabled=st.session_state.is_processing,
            help="The maximum number of input and output tokens that your model can use",
        )

        st.subheader("About")
        st.markdown("""
            PenTestPrompt streamlines the LLM security workflow by generating, executing, and evaluating attack prompts, significantly improving efficiency and coverage.
        """)

        return model_provider, selected_model, api_key, temperature, top_p, max_tokens