import json
import logging
import ndjson
import os
import re
import requests

from bs4 import BeautifulSoup
from typing import List

logger = logging.getLogger(__name__)

def extract_parameters(data):

    """
    Extracts relevant parameters from a given request data.

    Parameters:
    - data (str): A string containing request data. Typically, this includes headers, body, and other relevant information from an HTTP request.

    Returns:
    - status: The status of parameter extraction - whether it was successful or failed
    - url: The URL of the site
    - headers: The request headers
    - body: The request body

    Raises:
    - ValueError: If the provided data structure is not valid or lacks essential components for parameter extraction.

    """

    # Required format for output
    required_format = (
        "<METHOD> <FULL_URL> HTTP/<VERSION>\n\n"
        "Example: POST https://target.com/endpoint HTTP/2"
    )

    # Extracting API endpoint
    regex = r"^(POST|PUT|GET|DELETE|PATCH|OPTIONS|HEAD)\s+(https?://[a-zA-Z0-9.-]+(?:\:[0-9]+)?(?:/[^\s]*)?)\s+HTTP/(1\.0|1\.1|2)$"

    main_header = data[0].strip()
    logger.info("Extracting headers from the request file ...")
    if re.match(regex, main_header):
        method, url_endpoint, http_version = main_header.split(" ")
        if method not in ["POST", "PUT"]:
            return {"status": "error", "message": f"\n\nMethod must be POST or PUT\n\nExpected input format is: {required_format}"}
    else:
        return {"status": "error", "message": f"\n\nError occured while extracting headers from request file\n\nExpected input format is: {required_format}"}
    
    # Extracting Headers
    headers = {}
    line = 1
    while line < len(data):
        try:
            if data[line]=="\n":
                break

            key = data[line].split(":")[0]
            value = ':'.join(data[line].split(":")[1:]).strip()

        except Exception as e:
            logger.error(f"Error extracting headers in line {line}")
            return {"status": "error", "message": f"Error occured while extracting headers from request file\nError trace: {e}"}
        
        else:
            headers[key] = value
            line+=1

    logger.info("Extracting body from the request file ...")
    # Extracting body
    line+=1
    try:
        if (len(data)-line > 1): #Handling Multi-Line JSON
            body_str = ''.join(data[line:])
            body_json = json.loads(body_str)
            body = json.dumps(body_json, separators=(',', ':'))
        else:
            body = data[line]
        
        body = body.strip()

    except Exception as e:
        return {"status": "error", "message": f"Error occured while extracting body from request file\nError trace: {e}"}
    
    else:
        return {
            "status": "success", 
            "message": {
                "method": method,
                "url": url_endpoint,
                "headers": headers,
                "body": body
            }
        }

def get_response(request, prompt, special_token, method):

    if request['headers']['Content-Type']=="application/x-www-form-urlencoded":
        
        if special_token not in request['body']:
            return {"status": "error", "message": f"Request body doesn't contain special token {special_token}"}
        
        body = request["body"].replace(special_token, prompt).strip()
        request['headers']['Content-Length'] = str(len(body))
    
    elif request['headers']['Content-Type']=="application/json":
        
        if f'"{special_token}"' in request['body']:
            body = request["body"].replace(f'"{special_token}"', prompt)
        else:
            if f'{special_token}' in request['body']:
                body = request["body"].replace(special_token, prompt).strip()
            else:
                return {"status": "error", "message": f"Request body doesn't contain special token {special_token}"}

        request['headers']['Content-Length'] = str(len(body))

        #Converting to json_str
        x = json.loads(body)
        body = json.dumps(x)
    
    else:
        return {"status": "error", "message": "Only JSON and url-encoded requests are supported as input"}

    logger.info(f"Sending Request: {body}")

    if method == "POST":
        response = requests.post(url=request['url'], headers=request['headers'], data=body)
        # verify=False in above post request is must to ensure it does not give us SSLV3 handshake failure. 
        # Reference: https://stackoverflow.com/questions/18578439/using-requests-with-tls-doesnt-give-sni-support
        # Downgrading requests library to 2.28.2 and urllib3 to 1.26.2 will fix this
    
    elif method == "PUT":
        response = requests.put(url=request['url'], headers=request['headers'], data=body)

    else:
        return {"status": "error", "message": f"Method {method} not supported"}

    return response

def extract_response_text(response):
    """
    Args
    Input:
        response (dict): The response body returned on the POST request with updated prompt
    Output:
        status: indicating whether extracting text succeeded or failed
        text: If succeeded, the output text else the error message
    """
    if "Content-Type" in response.headers.keys():
        
        if "text/event-stream" in response.headers["Content-Type"]:
            logger.info("OUTPUT_TYPE: text/event-stream output")
            try:
                # Filtering output
                output = response.text.replace("data: ","").replace("\r","").strip().split("\n")
                output = [x for x in output if x]
                response_text = ""
                for chunk in output:
                    json_str = json.loads(chunk)
                    stream_segment = json_str['choices']['delta']['content']
                    if stream_segment is not None:
                        response_text = response_text + stream_segment + ""
            
            except Exception as e:
                return {"status": "error", "message": f"Error parsing event-stream output: {e}"}
            
            else:
                return {"status": "success", "message": response_text}
        
        elif "application/json" in response.headers["Content-Type"]:
            logger.info("OUTPUT_TYPE: JSON output")
            try:
                response_text = response.text
            
            except Exception as e:
                return {"status": "error", "message": f"Error parsing JSON output: {e}"}
            
            else:
                return {"status": "success", "message": response_text}
        
        elif "application/x-ndjson" in response.headers["Content-Type"]:
            logger.info("OUTPUT_TYPE: application/x-ndjson output")
            try:
                data = ndjson.loads(response.text)

                response_text = ""
                if data:
                    
                    last_object = data[-1]

                    if isinstance(last_object, dict):
                        if 'content' in last_object:
                            response_text = last_object.get('content', "")
                        elif 'message' in last_object:
                            response_text = last_object.get('message', "")
                        elif 'text' in last_object:
                            response_text = last_object.get('text', "")
                        elif 'value' in last_object:
                            response_text = last_object.get('value', "")

                    if not response_text:
                        response_text = json.dumps(last_object)

                if not response_text:
                    response_text = response

            except Exception as e:
                return {"status": "error", "message": f"Error processing application/x-ndjson output: {e}"}

            else:
                if not isinstance(response_text, str):
                    response_text = str(response_text)

                return {"status": "success", "message": response_text.strip()}
        
        elif "application/octet-stream" in response.headers["Content-Type"]:
            logger.info("OUTPUT_TYPE: application/octet-stream")
            
            try:
                response_text = response.text
                logger.info(f"Response text is : {response.text}")
                
            except Exception as e:
                return {"status": "error", "message": f"Error parsing application/octet-stream output: {e}"}
            
            else:
                return {"status": "success", "message": response_text}
        
        elif "text/html" in response.headers["Content-Type"]:
            logger.info("OUTPUT_TYPE: HTML output")
            try:
                soup = BeautifulSoup(response.text, 'html.parser')
                element = soup.find('body')
                text_content = element.get_text(separator=' ').strip()

            except Exception as e:
                return {"status": "error", "message": f"Error parsing HTML output: {e}"}
            
            else:
                return {"status": "success", "message": text_content}

        elif "text/plain" in response.headers["Content-Type"]:
            logger.info("OUTPUT_TYPE: plaintext output")
            try:
                response_text = response.text
            
            except Exception as e:
                return {"status": "error", "message": f"Error parsing plaintext output: {e}"}
            
            else:
                return {"status": "success", "message": response_text}
        
        else:
            return {"status": "error", "message": f"Content-Type is {response.headers['Content-Type']} not supported"}

    else:
        return {"status": "error", "message": f"Content-Type not found in Response Header: {response.headers}"}

def call_automate_requests(request_file: str, prompt: str, special_token: str, keywords: List[str]):
    """
    Function to automate the request-response handling of the generated prompts
    
    Args:
        request_file (str): The txt file containing the sample request
        prompt (str): a generated prompt
        special_token (str): Token to replace in request body
    """

    if not os.path.exists(request_file):
        raise Exception (f"Input File {request_file} does not exist")

    logger.info(f"Reading data from file {request_file}")
    with open(request_file,'r') as f:
        data = f.readlines()
    try:
        input_request_string = re.sub("###", prompt, "".join(data))
    except Exception as e:
        input_request_string = "".join(data)
    
    response_dict = {
        "raw_request_sent": input_request_string,
        "status": "",
        "status_code": None,
        "content_length": None,
        "response": "",
        "evaluation_status": False,
        "safe": None,
        "error_msg": "",
    }

    request_processor=extract_parameters(data)
    if request_processor["status"]=="error":
        response_dict["status"] = "error"
        response_dict["error_msg"] = f"Error extracting from request file: \n{request_processor['message']}"
        return response_dict

    response = None
    try:
        request = request_processor["message"]
        response = get_response(request, prompt, special_token, request["method"])
        logger.info(f"The response returned is {response.text}")
        
        content_length = response.headers.get('content_length', str(len(response.text)))
        response_msg = extract_response_text(response=response)
        cleaned_message = response_msg['message'].replace("\n", " ").replace("\t", " ").strip()
        cleaned_message = re.sub(r'\s+', ' ', cleaned_message)
        
        response_dict["status"] = "success"
        response_dict["status_code"] = response.status_code
        response_dict["content_length"] = content_length
        response_dict["response"] = cleaned_message
        
        if 200 <= response_dict["status_code"] < 300:
            if any(keyword.strip().lower() in cleaned_message.lower() for keyword in keywords):
                response_dict["safe"] = True
            else:
                response_dict["safe"] = False
            
            response_dict["evaluation_status"] = True
        
        else:
            response_dict["safe"] = None
            response_dict["evaluation_status"] = False
        return response_dict

    except Exception as e:
        response_dict["status_code"] = None
        if response:
            if "status_code" in response:
                response_dict["status_code"] = response.status_code
        response_dict["status"] = "error"
        response_dict["error_msg"] = f"Error in automated request-response: \n{str(e)}"
        
        return response_dict