import streamlit as st
from typing import Dict, List

def return_processing_overlay_style():
    return st.markdown(
        """
        <style>
            /* Main app container */
            .main > div:first-child {
                opacity: 0.7 !important;
                pointer-events: none !important;
            }

            /* Keeps the sidebar disabled */
            .stSidebar {
                opacity: 0.7 !important;
                pointer-events: none !important;
            }
            
            /* Keeps the radio button visible but disabled */
            .stRadio {
                opacity: 0.7 !important;
                pointer-events: none !important;
            }

            /* Keep the form elements visible but disabled */
            .stForm, .stDownloadButton {
                opacity: 0.7 !important;
                pointer-events: none !important;
            }

            /* For the Attack Type selection */
            [data-testid="stMultiSelect"] {
                opacity: 0.7 !important;
                pointer-events: none !important;
            }
            
            /* Keep attack prompts display visible */
            [data-testid="stText"] {
                opacity: 0.7 !important;
                pointer-events: none !important;
            }
            
            /* Spinner and progress elements */
            .stSpinner, .stProgress{
                opacity: 1 !important;
                pointer-events: auto !important;
                position: relative;
                z-index: 1000;
            }
        </style>
        """,
        unsafe_allow_html=True
    )

def reset_output_states():
    """Resets states related to output display."""
    st.session_state.automated_response_results = {}
    st.session_state.automated_evaluation_results = {}
    st.session_state.output_prompts = []
    st.session_state.error_message = None
    st.session_state.processing_complete = False

def _get_attack_prompts(attack_info: Dict, selected_attacks: List[str]) -> Dict[str, str]:
    """Get attack prompts for selected attacks"""
    attack_prompt_dic = {}
    
    for attack in selected_attacks:
        attack_prompt_dic[attack] = attack_info[attack]
    
    return attack_prompt_dic

def render_main_form(config: Dict) -> Dict:
    """Render the main form with generate/evaluate option"""
    applications = config.get('Application', {}).get('values', [])
    attack_info = config.get('Configuration', {}).get('attack_prompt', {})
    system_instruction = config.get('Configuration', {}).get('system_prompt', '')

    mode = st.radio(
        "Select Mode:",
        ["Generate Prompts", "Generate Prompts & Evaluate Results"],
        horizontal=True,
        key="mode_selection",
        disabled=st.session_state.is_processing,
        on_change=reset_output_states
    )

    selected_attacks = st.multiselect(
        "Select the Attack Type:",
        options=list(attack_info.keys()),
        key="attack_type",
        disabled=st.session_state.is_processing
    )
    st.session_state.selected_attacks = selected_attacks

    custom_prompt_input_value = ""

    with st.form(key="main_form"):
        col1, col2 = st.columns([1, 1])
        
        with col1:
            application = st.selectbox(
                "Select the Application Type:",
                options=applications,
                key="application",
                disabled=st.session_state.is_processing,
                index=None
            )

            other_app_value = st.session_state.get("application_other", "")
            if application == "Other":
                application = st.text_input(
                    "Specify the application type:",
                    value=other_app_value,
                    key="application_other",
                    disabled=st.session_state.is_processing
                )
            
            description = st.text_area(
                "Describe the Application Context:",
                key="description",
                disabled=st.session_state.is_processing,
                height=100
            )

            system_instruction_updated = st.text_area(
                label="System Instruction",
                value=system_instruction,
                key="system_instruction",
                disabled=st.session_state.is_processing,
                height=170
            )

            sample_request = None
            if mode == "Generate Prompts & Evaluate Results":
                
                sample_request = st.file_uploader(
                    "Upload Sample Request File:",
                    type="txt",
                    key="sample_request_uploader",
                    disabled=st.session_state.is_processing
                )

                st.markdown("""
                    :orange[**Note for Evaluation:**]
                    *   :red[The special token `###` must be present in the request body where prompts will be injected.]
                    *   :red[Required request format: `METHOD FULL_URL HTTP/VERSION` (e.g., `POST https://target.com/endpoint HTTP/2`)]
                    """)

        with col2:
                            
            attack_prompt_dic = _get_attack_prompts(attack_info, selected_attacks)
            if attack_prompt_dic:
                st.write("Selected Attack Instructions:")
            
            if "Custom Prompt" in selected_attacks:
                custom_prompt_input_value = st.text_area(
                    "Add Your Attack Prompt to generate more such prompts:",
                    key="custom_prompt_input",
                    disabled=st.session_state.is_processing,
                    height=220
                )
            
            if "Custom Prompt" in selected_attacks:
                attack_prompt_dic["Custom Prompt"] = custom_prompt_input_value

            for attack_name, attack_instruction in attack_prompt_dic.items():
                if attack_name != "Custom Prompt":
                    st.text_area(
                        label=f"Instruction for: {attack_name}",
                        value=attack_instruction,
                        key=f"display_{attack_name}",
                        disabled=st.session_state.is_processing,
                        height=120
                    )
            
            num_prompts = st.number_input(
                "Approximate Number of Prompts to Generate (per attack type):",
                min_value=1,
                value=config.get('Configuration', {}).get('num_prompts', 10),
                step=5,
                key="num_prompts",
                disabled=st.session_state.is_processing,
                help="How many prompt variations to generate for each selected attack type."
            )
            st.markdown(":orange[Recommendation: 5-50 prompts per attack type for balanced results.]")


            additional_prompts = None
            if mode == "Generate Prompts & Evaluate Results":
                additional_prompts = st.file_uploader(
                    "Upload Additional Prompts (Optional):",
                    type="txt",
                    key="additional_prompts_uploader",
                    disabled=st.session_state.is_processing,
                    help="Upload a .txt file with one prompt per line. These will be evaluated alongside any generated prompts."
                )
                st.markdown("""
                    :orange[These prompts will be evaluated directly. If you also provide generation details above, new prompts will be generated *and* added to these for evaluation.]""")

        button_label = "Generate Prompts" if mode == "Generate Prompts" else "Generate and Evaluate"
        
        submit_button = st.form_submit_button(
            button_label,
        )

        return {
            'mode': mode,
            'application': application,
            'description': description,
            'system_instruction': system_instruction_updated,
            'attack_prompt_dic': attack_prompt_dic,
            'num_prompts': num_prompts,
            'additional_prompts': additional_prompts,
            'sample_request': sample_request,
            'special_token': "###",
            'submit_button': submit_button,
        }