import json
import logging
import os
import pandas as pd
import sys
import yaml

from logging.handlers import RotatingFileHandler
from typing import Dict, Any, Union

_log_handlers = set()
logger = logging.getLogger(__name__)

def setup_logging(log_file: str = "log.log", log_level=logging.INFO):
    """Configures root logger. Ensures handlers are added only once."""
    global _log_handlers
    log_format = '%(asctime)s - %(process)d - %(name)s - %(levelname)s - %(message)s'
    date_format = '%Y-%m-%d %H:%M:%S'

    log_dir = os.path.join("Logs")
    os.makedirs(log_dir, exist_ok=True)
    log_path = os.path.join(log_dir, log_file)
    
    logger = logging.getLogger()
    logger.setLevel(log_level)

    handler_key_file = f"file_{log_path}"
    if handler_key_file not in _log_handlers:
        file_handler = RotatingFileHandler(
            log_path,
            maxBytes=100*1024*1024,
            backupCount=5,
            encoding='utf-8'
        )
        file_handler.setFormatter(logging.Formatter(log_format, date_format))
        logger.addHandler(file_handler)
        _log_handlers.add(handler_key_file)
        
    handler_key_console = "console_stdout"
    if handler_key_console not in _log_handlers:
        console_handler = logging.StreamHandler(sys.stdout)
        console_handler.setFormatter(logging.Formatter(log_format, date_format))
        logger.addHandler(console_handler)
        _log_handlers.add(handler_key_console)
    return logger

def load_yaml_config(yaml_path: str) -> Dict[str, Any]:
    """
        Load and parse the YAML configuration file.
    """
    try:
        logger.info(f"Loading configuration from {yaml_path}")
        with open(yaml_path, 'r', encoding='utf-8') as file:
            return yaml.load(file, Loader=yaml.SafeLoader)
    except FileNotFoundError:
        raise Exception(f"Error: Configuration file '{yaml_path}' not found.")
    except yaml.YAMLError as e:
        raise Exception(f"Error parsing YAML file: {e}")

def load_attack_config(attack_folder_path: str):
    """
    Parses through each YAML configuration file for Attacks
    Args:
        attack_folder_path (str): The folder path consisting of YAML configuration files for Attacks
    Returns:
        attack_dict (dict): A dictionary containing attack name along with corresponding prompts
    """
    attack_info = {}
    logger.info(f"Loading Attacks from {attack_folder_path}")
    for file in os.listdir(attack_folder_path):
        if file.endswith(".yaml"):
            yaml_path = os.path.join(attack_folder_path, file)
            attack = load_yaml_config(yaml_path=yaml_path)
            attack_info[attack['name']] = attack['value']

    return attack_info

def craft_input_prompt(application: str, application_description: str, attack_instruction: str, num_prompts: int) -> str:

    """
    Craft input prompt for the model

    Args:
        application (str): Application type
        application_description (str): Description of the application
        attack_instruction (str): The prompt to be placed in the user instruction
        num_prompts (int): Number of prompts to generate

    """
    try:
        prompt = f"""
            # APPLICATION OVERVIEW #
            The application is about {application}. The functionalities include {application_description}.
            #############
            # TECHNIQUE #
            {attack_instruction}
            #############
            # RESPONSE #
            Use the given TECHNIQUE and APPLICATION OVERVIEW to generate {num_prompts} prompts relevant to the described application above.
        """
        return prompt
    
    except Exception as e:
        raise ValueError(f"Error crafting input prompt: {str(e)}")

def process_response(response: Any) -> Dict[str, Union[str, list]]:
    
    msg = None
    try:
        logger.info("Processing response")
        
        # Checking the finish reason
        finish_reason = response.choices[0].finish_reason
        if finish_reason == "length":
            return {
                "status": "error",
                "message": "Reached Max Tokens limit. Model stopped generation prematurely. Consider increasing the Max Tokens setting."
            }
        if finish_reason == "content_filter":
            return {
                "status": "error",
                "message": "Input failed the Model's content filter. Please modify your prompt and try again."
            }

        msg = response.choices[0].message.content
        if not msg:
            return {
                "status": "error",
                "message": "Model returned empty content."
            }
        msg = msg.strip()
        msg = msg.replace('```json', '').replace('```', '').strip()

        json_data = json.loads(msg)
        prompts = json_data.get("prompts")

        if prompts is None:
            return {
                "status": "error",
                "message": f"Parsed JSON does not contain 'prompts' key. Parsed data: {json_data}"
            }
        
        if not isinstance(prompts, list):
            return {
                "status": "error",
                "message": f"'prompts' key does not contain a list. Found: {type(prompts)}. Parsed data: {json_data}"
            }
        
        return {
            "status": "success",
            "message": "Prompts Generated Successfully",
            "prompts": prompts
        }
    except AttributeError:
        logger.error(f"Error accessing attributes in model response structure: {response}", exc_info=True)
        return {
            "status": "error",
            "message": f"Error accessing attributes in model response structure. Response: {response}"
        }
    except json.JSONDecodeError as e:
        logger.error(f"Error parsing JSON response: {str(e)}. Cleaned content: {msg}", exc_info=True)
        return {
            "status": "error",
            "message": f"Error parsing JSON response: {str(e)}\nModel content (cleaned): {msg}"
        }
    except Exception as e:
        logger.error(f"Unexpected error processing response: {str(e)}. Response: {response}", exc_info=True)
        return {
            "status": "error",
            "message": f"Unexpected error processing response: {str(e)}\nModel content: {msg}"
        }

def dict_to_df(data_dict):
    """Convert nested dictionary to DataFrame with prompt as index"""
    # Create DataFrame from dict of dicts
    if not data_dict:
        logger.warning("No data to convert to DataFrame")
        return pd.DataFrame()
    
    logger.info("Converting dictionary to DataFrame")
    df = pd.DataFrame.from_dict(data_dict, orient='index')
    df.reset_index(inplace=True)
    df.rename(columns={'index': 'prompt'}, inplace=True)
    return df